/*global define */

define(["lib/Zoot", "src/math/Random"],
function (Z, Random) {
	"use strict";

	var kAbout = "$$$/private/animal/Behavior/Blinker/About=Auto Blinker, (c) 2014.";

	function chooseBlinkReplacements (self) {
		var closedB = self.closed;
		self.aBlinkLayers.forEach(function (lay) {
			if (closedB) {
				lay.trigger();
			}
		});
	}	

	function getGaussianRandomNumber (inMean, inStdev, inRandomGenerator) {
		var x, 					// x is a zero-mean normally distributed random number computed via Box-Muller transform
			y, 					// y is x transformed with specified mean and stdev
			r1 = 0, r2 = 0, 	// r1 and r2 are uniformly distributed random numbers
			epsilon = 0.00001;

		while (r1 * r2 < epsilon) {
			r1 = inRandomGenerator.random();
			r2 = inRandomGenerator.random();
		}

		x = Math.sqrt(-2.0 * Math.log(r1)) * Math.cos(2.0 * Math.PI * r2);

		y = inStdev * x + inMean;

		return y;
	}

	function getDuration (inClosed, inRate, inLength, inRandomness, inRandomGenerator) {
		var duration = 0, rateInS, lengthInS, stdev;

		rateInS = (inRate > 0) ? (60.0 / inRate) : -1.0;
		lengthInS = inLength / 1000.0;
		stdev = (inRandomness / 100.0) * rateInS;

		if (inClosed) {
			duration = getGaussianRandomNumber(lengthInS, 0.0, inRandomGenerator);
		}
		else {
			duration = getGaussianRandomNumber(rateInS, stdev, inRandomGenerator);
		}

		return duration;
	}

	return {
		about: 			kAbout,
		description: 	"$$$/animal/Behavior/Blinker/Desc=Trigger Blink-tagged layers regularly or randomly",
		uiName:  		"$$$/animal/Behavior/AutoBlinker/UIName=Auto Blink",
		defaultArmedForRecordOn: true,

		defineTags: function () {
			return {
				aTags: [
					{
						id: "Adobe.Face.LeftBlink",
						artMatches: ["left blink"],
						uiName: "$$$/animal/Behavior/Face/TagName/LeftBlink=Left Blink",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.Face"}]				

					},
					{
						id: "Adobe.Face.RightBlink",
						artMatches: ["right blink"],
						uiName: "$$$/animal/Behavior/Face/TagName/RightBlink=Right Blink",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.Face"}]								
					},
				]
			};
		},
		
		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			return [
				{ id: "rate", type: "slider", uiName: "$$$/animal/Behavior/Blinker/Parameter/rate=Blinks per Minute", precision: 1, dephault: 30, "min": 0, "max": 500 },
				{ id: "length", type: "slider", uiName: "$$$/animal/Behavior/Blinker/Parameter/length=Blink Duration", precision: 1, uiUnits: "ms", dephault: 80, "min": 0, "max": 1000 },
				{ id: "randomness", type: "slider", uiName: "$$$/animal/Behavior/Blinker/Parameter/randomness=Randomness", precision: 0, uiUnits: "%", dephault: 100, "min": 0, "max": 500 },
				{ id: "blinkLayers", type: "layer", uiName: "$$$/animal/Behavior/Blinker/Parameter/blinkLayers=Blink Layers", dephault: { match: "//Adobe.Face.LeftBlink|Adobe.Face.RightBlink"} }
			];
		},

		onCreateBackStageBehavior: function (self) {
			return { order: 0.15, importance : 0.0 };
		},

		onCreateStageBehavior: function (self, args) {
			// NOTE from WL: some options for how I would want to do this with a new API:
			//
			// 1)  - behavior gets some target puppets from Animal
			//     - if targets are not empty, just turn on/off targeted puppets as necessary
			//         = note that this means no "swapping" functionality, or we could by default turn off all other (non-target) siblings
			//     - if targets are empty, apply default matching rule (look for "Blink") to stagePuppet to identify target puppets to blink
			//
			// 2)  - behavior gets some target puppets from Animal
			//     - if targets are not empty, just turn on/off targeted puppets as necessary
			//         = note that this means no "swapping" functionality, or we could by default turn off all other (non-target) siblings
			//     - if targets are empty, do nothing
			//
			// option 1 is close to the functionality that we have right now, but we need target puppets from Animal.
			// option 2 makes the behavior simpler, but requires more manual work from user (identifying blink targets, even if they are named "Blink").
			// we could have option 3 where default matching rule is applied to each target puppet tree, but we lose generality (e.g., non-"Blink" puppets that we want to blink).

			var stagePuppet = args.stagePuppet, 
				allBlinkLayers = args.getStaticParam("blinkLayers");

			self.stagePuppet = stagePuppet;

			// for each blink puppet, get corresponding replacement group (siblings)
			self.aBlinkLayers = [];
			
			var bHideSiblings = true;

			allBlinkLayers.forEach(function (lay) {
				lay.setTriggerable(bHideSiblings);
				self.aBlinkLayers.push(lay);
			});

			// cache param values so we can detect when they change
			self.currentParams = {
				"rate": null,
				"length": null,
				"randomness": null
			};

			// initialize blink params
			this.onResetRehearsalData(self);
		},

		onResetRehearsalData : function (self) {
			var seed = 123456789;
			self.rand = new Random(seed);
			self.closed = false;
			self.duration = 0;		// duration of current state (open or closed)
			self.lastSwitchT = -1; 	// time of last switch (open or closed)
		},
		
		updateCurrentParams: function (self, args) {
			var changed = false, name, currentValue;

			for (name in self.currentParams) {
				currentValue = args.getParam(name);
				changed = changed || (self.currentParams[name] !== currentValue);
				self.currentParams[name] = currentValue;
			}

			return changed;
		},

		// TODO: duplicated in Adobe.Particles.js -- need to refactor!
		setSeedFromArgs : function (self, args) {
			var seed = args.t * 653 + args.globalRehearseTime * 109;
		
			// Mix in the behaviorInstanceId so we don't get repeated patterns for duplicated puppets.
			var idStr = args.stageLayer.getId();
			for (var i=0; i<idStr.length; ++i) {
				seed += idStr.charCodeAt(i);
			}
			self.rand.setSeed(seed);
		},

		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage	
			var paramsChanged, dt, t = args.t + args.globalRehearseTime;

			paramsChanged = this.updateCurrentParams(self, args);

			this.setSeedFromArgs(self, args);

			// if params have changed, allow new params to take effect immediately 
			// (rather than waiting until next blink)
			if (paramsChanged || self.lastSwitchT < 0) {
				self.lastSwitchT = t;
				self.duration = getDuration(self.closed, args.getParam("rate"), args.getParam("length"), args.getParam("randomness"), self.rand);
			}

			dt = t - self.lastSwitchT;

			if (dt >= self.duration) {
				self.lastSwitchT = t;
				self.closed = !self.closed;
				self.duration = getDuration(self.closed, args.getParam("rate"), args.getParam("length"), args.getParam("randomness"), self.rand);
			}

			chooseBlinkReplacements(self);
		}

	}; // end of object being returned
});
